<?php

class Byteonic_API
{
    private $option_name = 'byteonic_intake_api_key';

    public function get_api_key()
    {
        return get_option($this->option_name);
    }

    public function set_api_key($key)
    {
        update_option($this->option_name, sanitize_text_field($key));
    }

    public function delete_api_key()
    {
        delete_option($this->option_name);
    }

    /**
     * Make a request to the Byteonic Intake API
     */
    private function request($endpoint, $method = 'GET', $body = null, $custom_key = null)
    {
        $api_key = $custom_key ? $custom_key : $this->get_api_key();

        if (empty($api_key)) {
            return new WP_Error('missing_key', __('API Key is missing. Please configure it in the settings.', 'byteonic-intake'));
        }

        $url = BYTEONIC_API_BASE_URL . $endpoint;

        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[Byteonic API] Request URL: ' . $url);
        }

        $args = [
            'method' => $method,
            'headers' => [
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'User-Agent' => 'Byteonic-Intake-WP/1.0.0; ' . get_bloginfo('url'),
            ],
            'timeout' => 20,
        ];

        if ($body !== null) {
            $json_body = json_encode($body);
            if ($json_body === false) {
                return new WP_Error('json_error', 'JSON Encode Error: ' . json_last_error_msg());
            }
            $args['body'] = $json_body;
        }

        $response = wp_remote_request($url, $args);

        if (is_wp_error($response)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('[Byteonic API] Connection Error: ' . $response->get_error_message());
            }
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[Byteonic API] Response Code: ' . $code);
        }

        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);

        if ($code >= 400) {
            $error_msg = isset($data['error']) ? $data['error'] : __('Unknown API Error', 'byteonic-intake');

            // Append details if available (e.g. validation errors, limit reasons)
            if (!empty($data['details'])) {
                if (is_array($data['details'])) {
                    // Flatten array if needed, simplistic approach:
                    $details_str = [];
                    foreach ($data['details'] as $k => $v) {
                        $details_str[] = "$k: $v";
                    }
                    $error_msg .= ' (' . implode(', ', $details_str) . ')';
                } else {
                    $error_msg .= ' (' . $data['details'] . ')';
                }
            }

            return new WP_Error('api_error', $error_msg, ['status' => $code, 'data' => $data]);
        }

        return $data;
    }

    /**
     * Get list of forms
     */
    public function get_forms()
    {
        $response = $this->request('/forms');
        if (is_wp_error($response)) {
            return $response;
        }
        return isset($response['forms']) ? $response['forms'] : [];
    }

    /**
     * Get specific form details
     */
    public function get_form($slug)
    {
        $response = $this->request('/forms/' . $slug);
        if (is_wp_error($response)) {
            return $response;
        }
        return isset($response['form']) ? $response['form'] : null;
    }

    /**
     * Submit form data
     */
    public function submit_form($slug, $submission_data)
    {
        return $this->request('/forms/' . $slug . '/submit', 'POST', $submission_data);
    }

    /**
     * Test a specific API key (used during setup)
     */
    public function test_connection($key_to_test)
    {
        return $this->request('/forms', 'GET', null, $key_to_test);
    }
}
