<?php
use FluentForm\App\Helpers\Helper;

class Byteonic_Fluent_Forms
{
    private $api;

    public function __construct($api)
    {
        $this->api = $api;

        // Hook into Fluent Forms
        if (defined('FLUENTFORM') || class_exists('FluentForm\Framework\Foundation\Application')) {
            // Sidebar Menu
            add_filter('fluentform/form_settings_menu', [$this, 'add_settings_menu'], 10, 1);

            // Content Area - Hook into the main settings container to inject our tab content
            add_action('fluentform/form_settings_container_form_settings', [$this, 'render_settings_content'], 10, 1);

            // Submission Handler
            add_action('fluentform/submission_inserted', [$this, 'handle_submission'], 10, 3);
        }
    }

    private function log($message)
    {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[Byteonic Intake Fluent] ' . $message);
        }
    }

    public function add_settings_menu($menu_items)
    {
        $menu_items['byteonic_intake'] = [
            'title' => __('Byteonic Intake', 'byteonic-intake'),
            'slug'  => 'form_settings',
            'hash'  => 'byteonic-intake-settings',
            'route' => 'byteonic-intake-settings', // Triggers Hash-based nav (No Reload)
            'class' => 'ff-page-scroll',
        ];
        return $menu_items;
    }

    public function render_settings_content($form_id)
    {
        // Handle Saving
        if (isset($_POST['byteonic_save_settings']) && check_admin_referer('byteonic_fluent_save', 'byteonic_nonce')) {
            $settings = [
                'enabled' => isset($_POST['byteonic_enabled']),
                'form_slug' => sanitize_text_field($_POST['byteonic_form_slug']),
                'field_map' => isset($_POST['byteonic_field_map']) ? array_map('sanitize_text_field', $_POST['byteonic_field_map']) : []
            ];

            if (class_exists('\FluentForm\App\Helpers\Helper')) {
                Helper::setFormMeta($form_id, 'byteonic_intake_settings', $settings);
                echo '<div class="notice notice-success is-dismissible"><p>' . __('Settings saved.', 'byteonic-intake') . '</p></div>';
            }
        }

        // Fetch Settings
        $settings = [];
        if (class_exists('\FluentForm\App\Helpers\Helper')) {
            $settings = Helper::getFormMeta($form_id, 'byteonic_intake_settings');
        }

        $enabled = isset($settings['enabled']) ? $settings['enabled'] : false;
        $saved_slug = isset($settings['form_slug']) ? $settings['form_slug'] : '';
        $saved_map = isset($settings['field_map']) ? $settings['field_map'] : [];

        // Fetch API Forms
        $forms = $this->api->get_forms();

        // Fetch Fluent Form Fields for Mapping
        $fluent_fields = [];
        if (class_exists('\FluentForm\App\Helpers\Helper')) {
            $form = Helper::getForm($form_id);
            if ($form && isset($form->form_fields)) {
                $fields_data = json_decode($form->form_fields, true);
                $fluent_fields = $this->extract_fluent_fields($fields_data);
            }
        }

        ?>
        <div id="byteonic-intake-settings" class="ff_settings_section byteonic-fluent-wrapper" style="display: none;">
            <style>
                .byteonic-fluent-wrapper {
                    padding: 20px 0;
                    margin-bottom: 50px;
                    max-width: 800px;
                }
                .byteonic-card {
                    background: #fff;
                    border: 1px solid #e1e4e8;
                    border-radius: 4px;
                    padding: 24px;
                    box-shadow: 0 1px 3px rgba(0,0,0,0.04);
                }
                .byteonic-header {
                    margin-bottom: 24px;
                    border-bottom: 1px solid #eee;
                    padding-bottom: 16px;
                }
                .byteonic-header h2 {
                    margin: 0;
                    font-size: 18px;
                    font-weight: 600;
                    color: #2c3338;
                }
                .byteonic-form-group {
                    margin-bottom: 20px;
                }
                .byteonic-form-group label {
                    display: block;
                    font-weight: 600;
                    margin-bottom: 8px;
                    color: #333;
                }
                .byteonic-control {
                    width: 100%;
                    max-width: 400px;
                    padding: 8px 12px;
                    border: 1px solid #ccd0d4;
                    border-radius: 4px;
                    font-size: 14px;
                    color: #2c3338;
                }
                .byteonic-toggle {
                    display: flex;
                    align-items: center;
                    gap: 10px;
                }
                .byteonic-toggle input[type="checkbox"] {
                    margin: 0;
                    width: 18px;
                    height: 18px;
                }
                .byteonic-toggle span {
                    font-weight: 500;
                    font-size: 14px;
                }
                /* Mapping Table */
                .byteonic-mapping-table {
                    width: 100%;
                    border-collapse: separate;
                    border-spacing: 0;
                    border: 1px solid #e1e4e8;
                    border-radius: 6px;
                    overflow: hidden;
                    margin-top: 10px;
                }
                .byteonic-mapping-table th {
                    background: #f8f9fa;
                    padding: 12px 16px;
                    text-align: left;
                    font-weight: 600;
                    color: #555;
                    border-bottom: 1px solid #e1e4e8;
                }
                .byteonic-mapping-table td {
                    padding: 12px 16px;
                    border-bottom: 1px solid #eee;
                    vertical-align: middle;
                }
                .byteonic-mapping-table tr:last-child td {
                    border-bottom: none;
                }
                .byteonic-field-label {
                    font-weight: 500;
                    color: #2c3338;
                    display: block;
                }
                .byteonic-field-key {
                    color: #888;
                    font-size: 12px;
                    font-family: monospace;
                    display: block;
                    margin-top: 2px;
                }
                /* Button */
                .byteonic-actions {
                    margin-top: 30px;
                    padding-top: 20px;
                    border-top: 1px solid #eee;
                }
                .byteonic-btn-primary {
                    background-color: #409eff;
                    color: #fff;
                    border: none;
                    border-radius: 4px;
                    padding: 10px 24px;
                    font-size: 14px;
                    font-weight: 500;
                    cursor: pointer;
                    transition: background-color 0.2s;
                    text-decoration: none;
                    display: inline-block;
                }
                .byteonic-btn-primary:hover {
                    background-color: #66b1ff;
                    color: #fff;
                }
            </style>

            <form method="post" action="">
                <?php wp_nonce_field('byteonic_fluent_save', 'byteonic_nonce'); ?>
                <input type="hidden" name="byteonic_save_settings" value="1">

                <div class="byteonic-card">
                    <div class="byteonic-header">
                        <h2><?php _e('Byteonic Intake Configuration', 'byteonic-intake'); ?></h2>
                        <p style="margin: 5px 0 0; color: #666;"><?php _e('Connect this form to Byteonic Intake to automatically sync submissions.', 'byteonic-intake'); ?></p>
                    </div>

                    <div class="byteonic-form-group">
                        <label class="byteonic-toggle">
                            <input type="checkbox" name="byteonic_enabled" value="1" <?php checked($enabled, true); ?>>
                            <span><?php _e('Enable Integration for this Form', 'byteonic-intake'); ?></span>
                        </label>
                    </div>

                    <div class="byteonic-form-group">
                        <label><?php _e('Select Target Form', 'byteonic-intake'); ?></label>
                        <?php if (is_wp_error($forms)): ?>
                            <div class="notice notice-error inline" style="margin:0;"><p><?php echo esc_html($forms->get_error_message()); ?></p></div>
                        <?php else: ?>
                            <select name="byteonic_form_slug" class="byteonic-control">
                                <option value=""><?php _e('Select a form...', 'byteonic-intake'); ?></option>
                                <?php foreach ($forms as $form): ?>
                                    <option value="<?php echo esc_attr($form['slug']); ?>" <?php selected($saved_slug, $form['slug']); ?>>
                                        <?php echo esc_html($form['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <p class="description" style="margin-top:5px;"><?php _e('Choose the Byteonic form that should receive these submissions.', 'byteonic-intake'); ?></p>
                        <?php endif; ?>
                    </div>

                    <?php if ($saved_slug && !is_wp_error($forms)): ?>
                        <?php
                        $target_form = $this->api->get_form($saved_slug);
                        if ($target_form && !empty($target_form['fields'])):
                            ?>
                            <div class="byteonic-form-group" style="margin-top: 30px;">
                                <label style="margin-bottom: 15px;"><?php _e('Field Mapping', 'byteonic-intake'); ?></label>
                                
                                <table class="byteonic-mapping-table">
                                    <thead>
                                        <tr>
                                            <th style="width: 40%;"><?php _e('Byteonic Field', 'byteonic-intake'); ?></th>
                                            <th style="width: 60%;"><?php _e('Fluent Forms Field', 'byteonic-intake'); ?></th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($target_form['fields'] as $b_field):
                                            $b_key = $b_field['key'];
                                            $b_label = $b_field['label'];
                                            $required = !empty($b_field['required']) ? '<span style="color:#d63638" title="Required">*</span>' : '';
                                            $map_val = isset($saved_map[$b_key]) ? $saved_map[$b_key] : '';
                                            ?>
                                            <tr>
                                                <td>
                                                    <span class="byteonic-field-label"><?php echo esc_html($b_label); ?> <?php echo $required; ?></span>
                                                    <span class="byteonic-field-key"><?php echo esc_html($b_key); ?></span>
                                                </td>
                                                <td>
                                                    <select name="byteonic_field_map[<?php echo esc_attr($b_key); ?>]" class="byteonic-control" style="max-width: 100%;">
                                                        <option value=""><?php _e('Select Field...', 'byteonic-intake'); ?></option>
                                                        <?php foreach ($fluent_fields as $f_name => $f_label): ?>
                                                            <option value="<?php echo esc_attr($f_name); ?>" <?php selected($map_val, $f_name); ?>>
                                                                <?php echo esc_html($f_label); ?>
                                                            </option>
                                                        <?php endforeach; ?>
                                                    </select>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    <?php endif; ?>

                    <div class="byteonic-actions">
                        <button type="submit" class="byteonic-btn-primary"><?php _e('Save Integration Settings', 'byteonic-intake'); ?></button>
                    </div>
                </div>
            </form>
        </div>
        <script>
            jQuery(document).ready(function($) {
                // Logic to handle the Byteonic Intake settings tab
                // The Byteonic content is a sibling of the Vue .settings_app container
                // We need to hide .settings_app when showing our content, not .ff_settings_section
                
                function activateByteonicTab() {
                    // Hide the Fluent Forms Vue app container
                    $('.settings_app').hide();
                    // Show our section
                    $('#byteonic-intake-settings').show();
                    
                    // Manage Sidebar Active State
                    $('.ff_list_button_item').removeClass('ff_list_button_item_active');
                    $('.ff_list_button_link').removeClass('active');
                    
                    // Find our link and set active
                    $('a[href$="#byteonic-intake-settings"]').addClass('active').closest('li').addClass('ff_list_button_item_active');
                }
                
                function deactivateByteonicTab() {
                    // Hide our section
                    $('#byteonic-intake-settings').hide();
                    // Show the Fluent Forms Vue app container again
                    $('.settings_app').show();
                }

                // Listen for clicks on our menu item
                $(document).on('click', 'a[href$="#byteonic-intake-settings"]', function(e) {
                    e.preventDefault();
                    window.location.hash = 'byteonic-intake-settings';
                    activateByteonicTab();
                });
                
                // Listen for clicks on OTHER tabs - restore Fluent Forms Vue app
                $(document).on('click', '.ff_list_button_link:not([href$="#byteonic-intake-settings"])', function() {
                    deactivateByteonicTab();
                });

                // Check hash on load
                if (window.location.hash === '#byteonic-intake-settings') {
                    activateByteonicTab();
                }
            });
        </script>
        <?php
    }

    private function extract_fluent_fields($fields_data, &$results = [])
    {
        // Recursive extraction of fields from Fluent Form JSON structure (fields array)
        if (isset($fields_data['fields'])) {
            $fields_data = $fields_data['fields']; // Sometimes wrapper exists
        }

        foreach ($fields_data as $field) {
            // Check if container
            if (isset($field['fields']) && is_array($field['fields'])) {
                $this->extract_fluent_fields($field['fields'], $results);
            }
            // If field has name attribute, it's a field
            if (isset($field['attributes']['name']) && !empty($field['attributes']['name'])) {
                $label = isset($field['settings']['label']) ? $field['settings']['label'] : $field['attributes']['name'];
                $results[$field['attributes']['name']] = $label;
            }
        }
        return $results;
    }

    public function handle_submission($insertId, $submissionData, $form)
    {
        $form_id = $form->id;

        $settings = [];
        if (class_exists('\FluentForm\App\Helpers\Helper')) {
            $settings = Helper::getFormMeta($form_id, 'byteonic_intake_settings');
        }

        if (empty($settings) || empty($settings['enabled']) || empty($settings['form_slug'])) {
            return;
        }

        $slug = $settings['form_slug'];
        $field_map = isset($settings['field_map']) ? $settings['field_map'] : [];
        $data = [];

        foreach ($field_map as $b_key => $ff_name) {
            if (empty($ff_name))
                continue;

            if (isset($submissionData[$ff_name])) {
                $data[$b_key] = $submissionData[$ff_name];
            }
        }

        // Fallback auto-map
        if (empty($field_map) && empty($data)) {
            foreach ($submissionData as $key => $value) {
                // If key is simple word, nice. If complicated, might need sanitize.
                // Fluent keys are usually 'names[first_name]' or 'email'
                $clean_key = sanitize_title($key);
                $data[$clean_key] = $value;
            }
        }

        if (empty($data))
            return;

        // Meta
        $referer = wp_get_referer();
        $source_url = $referer ? $referer : home_url();

        $data['_meta'] = [
            'platform' => 'wordpress',
            'plugin' => 'fluent-forms',
            'form_id' => $form_id,
            'entry_id' => $insertId,
            'current_url' => $source_url
        ];

        $this->api->submit_form($slug, $data);
    }
}

