<?php

class Byteonic_Gravity_Forms {
    private $api;

    public function __construct( $api ) {
        $this->api = $api;
        $this->init();
    }

    public function init() {
        // Check if Gravity Forms is active
        if ( class_exists( 'GFForms' ) ) {
            // Add settings tab to Form Settings
            add_filter( 'gform_form_settings_menu', [ $this, 'add_settings_menu_item' ], 10, 2 );
            add_action( 'gform_form_settings_page_byteonic_intake', [ $this, 'render_settings_page' ] );
            
            // Handle Submission
            add_action( 'gform_after_submission', [ $this, 'handle_submission_hook' ], 10, 2 );
        }
    }




    public function add_settings_menu_item( $menu_items, $form_id ) {
        $menu_items[] = [
            'name' => 'byteonic_intake',
            'label' => __( 'Byteonic Intake', 'byteonic-intake' )
        ];
        return $menu_items;
    }

    /**
     * Render the settings page for a specific form
     */
    public function render_settings_page() {
        $form_id = rgget( 'id' );
        $form = GFAPI::get_form( $form_id );
        
        // Save Settings
        if ( isset( $_POST['byteonic_save_settings'] ) ) {
            check_admin_referer( 'byteonic_gf_save', 'byteonic_gf_nonce' );
            $settings = [
                'enabled' => isset( $_POST['byteonic_enabled'] ),
                'form_slug' => sanitize_text_field( $_POST['byteonic_form_slug'] ),
                'field_map' => isset( $_POST['byteonic_field_map'] ) ? array_map( 'sanitize_text_field', $_POST['byteonic_field_map'] ) : []
            ];
            
            // Save to form meta
            gform_update_meta( $form_id, 'byteonic_intake_settings', $settings );
            echo '<div class="updated notice is-dismissible"><p>' . __( 'Settings saved.', 'byteonic-intake' ) . '</p></div>';
        }

        // Get Saved Settings
        $settings = gform_get_meta( $form_id, 'byteonic_intake_settings' );
        $enabled = isset( $settings['enabled'] ) ? $settings['enabled'] : false;
        $saved_slug = isset( $settings['form_slug'] ) ? $settings['form_slug'] : '';
        $field_map = isset( $settings['field_map'] ) ? $settings['field_map'] : [];

        // Fetch available forms from API
        $available_forms = $this->api->get_forms();

        GFFormSettings::page_header();
        ?>
        <div style="display: flex; align-items: center; margin-bottom: 20px; border-bottom: 1px solid #eee; padding-bottom: 15px;">
            <h2 style="margin:0; color: #4F46E5;"><?php _e( 'Byteonic Intake Integration', 'byteonic-intake' ); ?></h2>
            <span style="margin-left: 10px; font-size: 12px; background: #e0e7ff; color: #3730a3; padding: 2px 8px; border-radius: 12px; border: 1px solid #c7d2fe;">v<?php echo BYTEONIC_INTAKE_VERSION; ?></span>
        </div>
        <form method="post">
            <?php wp_nonce_field( 'byteonic_gf_save', 'byteonic_gf_nonce' ); ?>
            <table class="gforms_form_settings" cellspacing="0" cellpadding="0">
                <tr>
                    <th><label for="byteonic_enabled"><?php _e( 'Enable Integration', 'byteonic-intake' ); ?></label></th>
                    <td>
                        <input type="checkbox" id="byteonic_enabled" name="byteonic_enabled" value="1" <?php checked( $enabled, true ); ?> />
                        <label for="byteonic_enabled"><?php _e( 'Send submissions to Byteonic Intake', 'byteonic-intake' ); ?></label>
                    </td>
                </tr>
                <tr>
                    <th><label for="byteonic_form_slug"><?php _e( 'Select Target Form', 'byteonic-intake' ); ?></label></th>
                    <td>
                        <?php if ( is_wp_error( $available_forms ) ): ?>
                            <div style="color:red;"><?php echo $available_forms->get_error_message(); ?></div>
                        <?php else: ?>
                            <select name="byteonic_form_slug" id="byteonic_form_slug">
                                <option value=""><?php _e( 'Select a form...', 'byteonic-intake' ); ?></option>
                                <?php foreach ( $available_forms as $byteonic_form ): ?>
                                    <option value="<?php echo esc_attr( $byteonic_form['slug'] ); ?>" <?php selected( $saved_slug, $byteonic_form['slug'] ); ?>>
                                        <?php echo esc_html( $byteonic_form['name'] ); ?> (<?php echo esc_html( $byteonic_form['slug'] ); ?>)
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        <?php endif; ?>
                    </td>
                </tr>
            </table>

            <?php if ( $saved_slug && ! is_wp_error( $available_forms ) ): 
                // Get details for the selected form to show mapping
                $target_form_details = $this->api->get_form( $saved_slug );
            ?>
                <?php if ( $target_form_details && ! empty( $target_form_details['fields'] ) ): ?>
                    <h4><?php _e( 'Field Mapping', 'byteonic-intake' ); ?></h4>
                    <p><?php _e( 'Map your Gravity Form fields to Byteonic Intake fields.', 'byteonic-intake' ); ?></p>
                    <table class="widefat fixed" style="width: auto; min-width: 500px;">
                        <thead>
                            <tr>
                                <th><?php _e( 'Byteonic Field', 'byteonic-intake' ); ?></th>
                                <th><?php _e( 'Gravity Form Field', 'byteonic-intake' ); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ( $target_form_details['fields'] as $b_field ): ?>
                                <tr>
                                    <td>
                                        <strong><?php echo esc_html( $b_field['label'] ); ?></strong>
                                        <?php if ( ! empty( $b_field['required'] ) ) echo ' <span style="color:red">*</span>'; ?>
                                        <br><small>Key: <?php echo esc_html( $b_field['key'] ); ?></small>
                                    </td>
                                    <td>
                                        <select name="byteonic_field_map[<?php echo esc_attr( $b_field['key'] ); ?>]">
                                            <option value=""><?php _e( 'Select user input...', 'byteonic-intake' ); ?></option>
                                            <?php foreach ( $form['fields'] as $gf_field ): 
                                                if ( $gf_field->displayOnly ) continue;
                                                $map_value = isset( $field_map[ $b_field['key'] ] ) ? $field_map[ $b_field['key'] ] : '';
                                            ?>
                                                <option value="<?php echo esc_attr( $gf_field->id ); ?>" <?php selected( $map_value, $gf_field->id ); ?>>
                                                    <?php echo esc_html( $gf_field->label ); ?> (ID: <?php echo esc_html( $gf_field->id ); ?>)
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            <?php endif; ?>

            <p class="submit">
                <input type="submit" name="byteonic_save_settings" class="button button-primary" value="<?php _e( 'Save Integration Settings', 'byteonic-intake' ); ?>">
            </p>
        </form>
        <?php
        GFFormSettings::page_footer();
    }

    /**
     * Handle the submission
     */
    public function handle_submission_hook( $entry, $form ) {
        $form_id = $form['id'];
        $settings = gform_get_meta( $form_id, 'byteonic_intake_settings' );

        if ( empty( $settings ) || empty( $settings['enabled'] ) || empty( $settings['form_slug'] ) ) {
            return;
        }

        $slug = $settings['form_slug'];
        $field_map = isset( $settings['field_map'] ) ? $settings['field_map'] : [];
        $data = [];

        // Process Mapping
        foreach ( $field_map as $byteonic_key => $gf_field_id ) {
            if ( empty( $gf_field_id ) ) {
                continue;
            }

            // robustly retrieve value using GF helper rgar which handles various entry formats
            $value = rgar( $entry, $gf_field_id );

            if ( $value !== null && $value !== '' ) {
                $data[ $byteonic_key ] = $value;
            }
        }
        
        // Add Meta
        $data['_meta'] = [
            'current_url' => rgar( $entry, 'source_url' ),
            'remote_ip'   => rgar( $entry, 'ip' ),
            'user_agent'  => rgar( $entry, 'user_agent' ),
            'plugin'      => 'gravity-forms',
            'version'     => BYTEONIC_INTAKE_VERSION
        ];

        GFCommon::log_debug( "Byteonic Intake: Preparing to submit data for Entry {$entry['id']}. Data: " . print_r( $data, true ) );
        
        // Always try to find email if mapped, or auto-detect if not mapped (optional heuristic)
        // For now strict mapping is safer.

        // Send to API
        $result = $this->api->submit_form( $slug, $data );

        if ( is_wp_error( $result ) ) {
            GFCommon::log_debug( "Byteonic Intake: Submission failed for entry {$entry['id']}. Error: " . $result->get_error_message() );
        } else {
            GFCommon::log_debug( "Byteonic Intake: Submission successful for entry {$entry['id']}." );
        }
    }
}
